implementation module controlresize


//	Clean Object I/O library, version 1.0.1


import StdBool, StdFunc, StdInt, StdList, StdMisc, StdTuple
import commondef, controllayout, windowclipstate, windowdefaccess, windowhandle, wstateaccess


controlresizeFatalError :: String String -> .x
controlresizeFatalError rule error
	= FatalError rule "controlresize" error


/*	resizeControls proceeds as follows:
	-	Apply to every control its ControlResizeFunction if applicable.
	-	If some controls have changed size or have been layout relative to the window view frame:
		-	Calculate the new layout.
		-	Reposition and resize the appropriate controls.
*/
resizeControls :: OSWindowMetrics !ViewFrame !ViewFrame !(WindowStateHandle .ps) !*OSToolbox -> (!WindowStateHandle .ps,!*OSToolbox)
resizeControls wMetrics oldFrame newFrame wsH=:{wshIds={wPtr},wshHandle=Just wlsH=:{wlsHandle=wH=:{whWindowInfo,whItems,whAtts}}} tb
	# (oldItems`,oldItems,tb)	= getWElementHandles` wPtr whItems tb
	  (layoutChanged,newItems`)	= calcNewControlsSize wMetrics oldWSize newWSize oldItems`
	| not layoutChanged
	= ({wsH & wshHandle=Just {wlsH & wlsHandle={wH & whItems=oldItems}}},tb)
	# newItems					= setWElementHandles` newItems` oldItems
	# (_,newItems,tb)			= layoutControls wMetrics hMargins vMargins spaces newWSize minSize domain.corner1 origin newItems tb
	# (newItems`,newItems,tb)	= getWElementHandles` wPtr newItems tb
	# tb						= relayoutControls wMetrics (0,0) wPtr oldItems` newItems` tb
	= ({wsH & wshHandle=Just {wlsH & wlsHandle={wH & whItems=newItems}}},tb)
where
	oldWSize					= rectangleSize oldFrame
	newWSize					= rectangleSize newFrame
	windowInfo					= fromJust whWindowInfo
	origin						= windowInfo.windowOrigin
	domain						= windowInfo.windowDomain
	(defHMargin,defVMargin)		= (wMetrics.osmHorMargin,wMetrics.osmVerMargin)
	(defMinW,  defMinH)			= OSMinWindowSize
	(defHSpace,defVSpace)		= (wMetrics.osmHorItemSpace,wMetrics.osmVerItemSpace)
	hMargins					= getwindowhmargin     (snd (Select iswindowhmargin     (WindowHMargin defHMargin defHMargin) whAtts))
	vMargins					= getwindowvmargin     (snd (Select iswindowvmargin     (WindowVMargin defVMargin defVMargin) whAtts))
	spaces						= getwindowitemspace   (snd (Select iswindowitemspace   (WindowItemSpace   defHSpace defVSpace) whAtts))
	minSize						= getwindowminimumsize (snd (Select iswindowminimumsize (WindowMinimumSize {w=defMinW,h=defMinH}) whAtts))
resizeControls _ _ _ _ _
	= controlresizeFatalError "resizeControls" "unexpected window placeholder argument"


/*	calcNewControlsSize applies to a Control its ControlResizeFunction if it has one.
	The Boolean result holds iff some Control has changed its size or may cause change of layout.
*/
calcNewControlsSize :: OSWindowMetrics !Size !Size ![WElementHandle`] -> (!Bool,![WElementHandle`])
calcNewControlsSize wMetrics oldWSize newWSize itemHs
	| isEmpty itemHs
	= (False,itemHs)
	# (itemH,itemHs)			= HdTl itemHs
	  (layoutChanged1,itemH)	= calcNewControlSize  wMetrics oldWSize newWSize itemH
	  (layoutChanged2,itemHs)	= calcNewControlsSize wMetrics oldWSize newWSize itemHs
	= (layoutChanged1 || layoutChanged2,[itemH:itemHs])
where
	calcNewControlSize :: OSWindowMetrics !Size !Size !WElementHandle` -> (!Bool,!WElementHandle`)
	calcNewControlSize wMetrics oldWSize newWSize (WRecursiveHandle` itemHs wRecKind)
		# (layoutChanged,itemHs)= calcNewControlsSize wMetrics oldWSize newWSize itemHs
		= (layoutChanged,WRecursiveHandle` itemHs wRecKind)
	calcNewControlSize wMetrics oldWSize newWSize itemH`=:(WItemHandle` itemH)
		| not resizable
		= (isViewFrameSensitive,itemH`)
		# (layoutChanged,itemH)	= calcNewWItemSize wMetrics (\oldCSize -> resizeF oldCSize oldWSize newWSize) itemH
		= (isViewFrameSensitive || layoutChanged,WItemHandle` itemH)
	where
		atts					= itemH.wItemAtts`
		(resizable,resizeAtt)	= Select isControlResize` undef atts
		(hasPos,posAtt)			= Select isControlPos` undef atts
		resizeF					= getControlResize` resizeAtt
		isViewFrameSensitive	= if hasPos
									(case (fst (getControlPos` posAtt)) of
										LeftBottom	-> True
										RightTop	-> True
										RightBottom	-> True
										Center		-> True
										Right		-> True
										_			-> False
									)
									False
		
		calcNewWItemSize :: !OSWindowMetrics !(IdFun Size) !WItemHandle` -> (!Bool,!WItemHandle`)
		calcNewWItemSize _ resizeF itemH=:{wItemKind`=IsCustomButtonControl}
			# itemH		= {itemH & wItemSize`=newSize1,wItemAtts`=replaceSizeAtt newSize itemH.wItemAtts`}
			= (newSize1<>oldSize,itemH)
		where
			oldSize		= itemH.wItemSize`
			newSize		= resizeF oldSize
			newSize1	= {w=max 0 newSize.w,h=max 0 newSize.h}
		
		calcNewWItemSize wMetrics resizeF itemH=:{wItemKind`=IsEditControl}
			# itemH		= {itemH & wItemSize`=newSize1,wItemInfo`=editInfo,wItemAtts`=replaceSizeAtt newSize1 itemH.wItemAtts`}
			= (newSize1<>oldSize,itemH)
		where
			oldSize		= itemH.wItemSize`
			newSize		= resizeF oldSize
			info		= getWItemEditInfo` itemH.wItemInfo`
			lineHeight	= wMetrics.osmHeight
			nrLines1	= max 1 (newSize.h/lineHeight)
			newSize1	= {w=max 0 newSize.w,h=nrLines1*lineHeight}
			editInfo	= EditInfo` {info & editInfoWidth=newSize1.w,editInfoNrLines=nrLines1}
		
		calcNewWItemSize _ resizeF itemH=:{wItemKind`=IsCustomControl}
			# itemH		= {itemH & wItemSize`=newSize1,wItemAtts`=replaceSizeAtt newSize1 itemH.wItemAtts`}
			= (newSize1<>oldSize,itemH)
		where
			oldSize		= itemH.wItemSize`
			newSize		= resizeF oldSize
			newSize1	= {w=max 0 newSize.w,h=max 0 newSize.h}
		
		calcNewWItemSize wMetrics resizeF itemH=:{wItemKind`=IsSliderControl}
			# itemH		= {itemH & wItemSize`=newSize1,wItemInfo`=sliderInfo,wItemAtts`=replaceSizeAtt newSize1 itemH.wItemAtts`}
			= (newSize1<>oldSize,itemH)
		where
			oldSize		= itemH.wItemSize`
			newSize		= resizeF oldSize
			info		= getWItemSliderInfo` itemH.wItemInfo`
			horizontal	= info.sliderInfoDir`==Horizontal
			newSize1	= if horizontal	{w=max newSize.w 0,h=wMetrics.osmHSliderHeight} {w=wMetrics.osmVSliderWidth,h=max newSize.h 0}
			sSize		= if horizontal newSize1.w newSize1.h
			sliderInfo	= SliderInfo` {info & sliderInfoLength`=sSize}
		
		calcNewWItemSize wMetrics resizeF itemH=:{wItemKind`=IsCompoundControl}
			# (visHScroll,visVScroll)	= OSscrollbarsAreVisible wMetrics domainRect (SizeToTuple itemSize) (hasHScroll,hasVScroll)
			  contentRect				= getCompoundContentRect wMetrics visHScroll visVScroll (PosSizeToRect itemPos itemSize)
			  oldSize					= RectSize contentRect
			  newSize					= resizeF oldSize
			  newSize					= {w=max minSize.w newSize.w,h=max minSize.h newSize.h}
			| newSize==oldSize
			= (False,itemH)
			# (newW,newH)				= SizeToTuple newSize
			  newOrigin					= newOrigin origin domainRect (newW,newH)
			  newInfo					= CompoundInfo` {info & compoundOrigin=newOrigin}
			  visHScroll				= hasHScroll && (OSscrollbarIsVisible (domain.corner1.x,domain.corner2.x) newW)
			  visVScroll				= hasVScroll && (OSscrollbarIsVisible (domain.corner1.y,domain.corner2.y) newH)
			  (newW`,newH`)				= (newW+wMetrics.osmVSliderWidth,newH+wMetrics.osmHSliderHeight)
			  newItemSize				= if (visHScroll && visVScroll) {w=newW`,h=newH`}
										 (if  visHScroll				{newSize & h=newH`}
										 (if  visVScroll				{newSize & w=newW`}
										 								newSize))
			# (_,itemHs)				= calcNewControlsSize wMetrics oldSize newSize itemH.wItems`
			  itemH						= {itemH & wItemSize`=newItemSize,wItemAtts`=replaceSizeAtt newSize atts,wItemInfo`=newInfo,wItems`=itemHs}
			  itemH						= invalidateCompoundClipState` itemH
			= (True,itemH)
		where
			atts						= itemH.wItemAtts`
			itemPos						= itemH.wItemPos`
			itemSize					= itemH.wItemSize`
			info						= getWItemCompoundInfo` itemH.wItemInfo`
			origin						= info.compoundOrigin
			domain						= info.compoundDomain
			domainRect					= RectangleToRect domain
			hasHScroll					= isJust info.compoundHScroll
			hasVScroll					= isJust info.compoundVScroll
			(defMinW,defMinH)			= OSMinCompoundSize
			minSize						= getControlMinimumSize` (snd (Select isControlMinimumSize` (ControlMinimumSize` {w=defMinW,h=defMinH}) atts))
			
			newOrigin :: !Point !Rect !(!Int,!Int) -> Point		// This code also appears at windowdevice: windowStateSizeAction
			newOrigin {x,y} (xMin,yMin,xMax,yMax) (w,h)
				= {x=x`,y=y`}
			where
				x`	= if (x+w>xMax) (max (xMax-w) xMin) x
				y`	= if (y+h>yMax) (max (yMax-h) yMin) y
		
		calcNewWItemSize _ _ itemH
			= (False,itemH)
	
	replaceSizeAtt :: !Size ![ControlAttribute`] -> [ControlAttribute`]
	replaceSizeAtt size atts
		# (replaced,atts)	= Replace isControlSize` sizeAtt atts
		| replaced
			= atts
			= [sizeAtt:atts]
	where
		sizeAtt = ControlSize` size


/*	relayoutControls resizes and moves changed WElementHandles.
	The first argument is the position of the parent window/compound.
	The OSWindowPtr is the parent window/dialog.
	The first  list contains the elements at their original location and size.
	The second list contains the elements at their new location and size.
	relayoutControls assumes that the two lists contain elements that are identical except for size and position.
*/
relayoutControls :: OSWindowMetrics !(!Int,!Int) !OSWindowPtr ![WElementHandle`] ![WElementHandle`] !*OSToolbox -> *OSToolbox
relayoutControls wMetrics parentPos wPtr [oldH:oldHs] [newH:newHs] tb
	# tb	= relayoutControl  wMetrics parentPos wPtr oldH  newH  tb
	# tb	= relayoutControls wMetrics parentPos wPtr oldHs newHs tb
	= tb
where
	relayoutControl :: OSWindowMetrics !(!Int,!Int) !OSWindowPtr !WElementHandle` !WElementHandle` !*OSToolbox -> *OSToolbox
	relayoutControl wMetrics parentPos wPtr (WRecursiveHandle` oldHs oldWRecKind) (WRecursiveHandle` newHs newWRecKind) tb
		| oldWRecKind==newWRecKind
		= relayoutControls wMetrics parentPos wPtr oldHs newHs tb
	relayoutControl wMetrics parentPos wPtr (WItemHandle` oldH=:{wItemKind`}) (WItemHandle` newH) tb
		| oldH.wItemKind`<>newH.wItemKind`
		= controlresizeFatalError "relayoutControl" "mismatching WItemHandle`"
		| oldH.wItemPos`==newH.wItemPos` && oldH.wItemSize`==newH.wItemSize`
		= tb
		= relayoutWItemHandle` wMetrics parentPos wPtr wItemKind` oldH newH tb
	where
		/*	relayoutWItemHandle` assumes that the two WItemHandle` arguments 
			have the same ControlKind (third argument) and differ in size or position or both.
		*/
		relayoutWItemHandle` :: OSWindowMetrics !(!Int,!Int) !OSWindowPtr !ControlKind !WItemHandle` !WItemHandle` !*OSToolbox -> *OSToolbox
		relayoutWItemHandle` wMetrics parentPos wPtr IsRadioControl old new tb
			= StateMap2 move info.radioItems` tb
		where
			info	= getWItemRadioInfo` new.wItemInfo`
			move {radioItemPos`,radioItemSize`,radioItemPtr`} tb
					= OSsetRadioControlPos wPtr parentPos radioItemPtr` (PointToTuple radioItemPos`) (SizeToTuple radioItemSize`) tb
		
		relayoutWItemHandle` wMetrics parentPos wPtr IsCheckControl old new tb
			= StateMap2 move info.checkItems` tb
		where
			info	= getWItemCheckInfo` new.wItemInfo`
			move {checkItemPos`,checkItemSize`,checkItemPtr`} tb
					= OSsetCheckControlPos wPtr parentPos checkItemPtr` (PointToTuple checkItemPos`) (SizeToTuple checkItemSize`) tb
		
		relayoutWItemHandle` wMetrics parentPos wPtr IsCompoundControl old new tb
			| oldSize==newSize
				# (moveItems,tb)	= OSsetCompoundPos wPtr parentPos itemPtr newPos newSize tb
				| moveItems
					= relayoutControls wMetrics newPos wPtr old.wItems` new.wItems` tb
					= tb
			# (_,tb)				= OSsetCompoundPos  wPtr parentPos itemPtr newPos oldSize tb
			# tb					= OSsetCompoundSize wPtr parentPos itemPtr newPos newSize tb
			# tb					= setCompoundScrollThumbValues hasHScroll itemPtr True  hThumbSize oldOrigin.x newOrigin.x tb
			# tb					= setCompoundScrollThumbValues hasVScroll itemPtr False vThumbSize oldOrigin.y newOrigin.y tb
			= relayoutControls wMetrics newPos wPtr old.wItems` new.wItems` tb
		where
			itemPtr					= new.wItemPtr`
			oldSize					= SizeToTuple  old.wItemSize`
			newPos					= PointToTuple new.wItemPos`
			newSize					= SizeToTuple  new.wItemSize`
			(newW,newH)				= newSize
			oldInfo					= getWItemCompoundInfo` old.wItemInfo`
			oldOrigin				= oldInfo.compoundOrigin
			newInfo					= getWItemCompoundInfo` new.wItemInfo`
			newOrigin				= newInfo.compoundOrigin
			domain					= newInfo.compoundDomain
			domainRect				= RectangleToRect domain
			hasHScroll				= isJust newInfo.compoundHScroll
			hasVScroll				= isJust newInfo.compoundVScroll
			(visHScroll,visVScroll)	= OSscrollbarsAreVisible wMetrics domainRect newSize (hasHScroll,hasVScroll)
			hThumbSize				= if visVScroll (newW-wMetrics.osmVSliderWidth +1) (newW+1)
			vThumbSize				= if visHScroll (newH-wMetrics.osmHSliderHeight+1) (newH+1)
			
			setCompoundScrollThumbValues :: !Bool OSWindowPtr Bool Int Int Int !*OSToolbox -> *OSToolbox
			setCompoundScrollThumbValues hasScroll compoundPtr isHorizontal size old new tb
				| not hasScroll
				= tb
				# tb				= OSsetCompoundSliderThumbSize compoundPtr isHorizontal size (old==new) tb
				| old==new
				= tb
				= OSsetCompoundSliderThumb compoundPtr isHorizontal new True tb
		
		relayoutWItemHandle` wMetrics parentPos wPtr controlKind old new tb
			# tb	= moveF tb
			# tb	= sizeF tb
			= tb
		where
			itemPtr				= new.wItemPtr`
			oldPos				= PointToTuple old.wItemPos`
			oldSize				= SizeToTuple  old.wItemSize`
			newPos				= PointToTuple new.wItemPos`
			newSize				= SizeToTuple  new.wItemSize`
			(setPos,setSize)	= case controlKind of
									IsPopUpControl			-> (OSsetPopUpControlPos,		OSsetPopUpControlSize)
									IsSliderControl			-> (OSsetSliderControlPos,		OSsetSliderControlSize)
									IsTextControl			-> (OSsetTextControlPos,		OSsetTextControlSize)
									IsEditControl			-> (OSsetEditControlPos,		OSsetEditControlSize)
									IsButtonControl			-> (OSsetButtonControlPos,		OSsetButtonControlSize)
									IsCustomButtonControl	-> (OSsetCustomButtonControlPos,OSsetCustomButtonControlSize)
									IsCustomControl			-> (OSsetCustomControlPos,		OSsetCustomControlSize)
									(IsOtherControl _)		-> (\_ _ _ _ _ tb->tb,			\_ _ _ _ _ tb->tb)
									_						-> controlresizeFatalError "relayoutWItemHandle`" "unexpected ControlKind alternative"
			moveF				= if (oldPos ==newPos ) id (setPos  wPtr parentPos itemPtr newPos oldSize)
			sizeF				= if (oldSize==newSize) id (setSize wPtr parentPos itemPtr newPos newSize)
		
	relayoutControl _ _ _ _ _ _
		= controlresizeFatalError "relayoutControl" "mismatching WElementHandle`"
relayoutControls _ _ _ [] [] tb
	= tb
relayoutControls _ _ _ _ _ _
	= controlresizeFatalError "relayoutControls" "mismatching WElementHandle`s"

/* Mac oswindow version.
OSsetEditControlPos :: !OSWindowPtr !(!Int,!Int) !OSWindowPtr !(!Int,!Int) !(!Int,!Int) !*OSToolbox -> *OSToolbox
OSsetEditControlPos parentWindow (parent_x,parent_y) hTE (x,y) (w,h) tb
	# (tePtr,tb)	= LoadLong hTE tb
	# tb			= StoreRect (tePtr+destRectOffset) newRect tb	// Directly write destination field for moving
	# tb			= StoreRect (tePtr+viewRectOffset) newRect tb	// Directly write view        field for moving
	# tb			= TECalText hTe tb
	= tb
where
	destRectOffset	= 0
	viewRectOffset	= 8
	newRect			= (x,y, x+w,y+h)

OSsetEditControlSize :: !OSWindowPtr !(!Int,!Int) !OSWindowPtr !(!Int,!Int) !(!Int,!Int) !*OSToolbox -> *OSToolbox
OSsetEditControlSize parentWindow (parent_x,parent_y) hTE (x,y) (w,h) tb
	# (tePtr,tb)	= LoadLong hTE tb
	# tb			= StoreRect (tePtr+destRectOffset) newRect tb	// Directly write destination field for sizing
	# tb			= StoreRect (tePtr+viewRectOffset) newRect tb	// Directly write view        field for sizing
	# tb			= TECalText hTe tb
	= tb
where
	destRectOffset	= 0
	viewRectOffset	= 8
	newRect			= (x,y, x+w,y+h)
*/
